package com.vmware.samples.chassisRackVSphere;

import java.net.URI;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.vmware.samples.chassisRackVSphere.ObjectStore;
import com.vmware.samples.chassisRackVSphere.model.ChassisInfo;
import com.vmware.samples.chassisRackVSphere.model.Chassis;
import com.vmware.samples.chassisRackVSphere.model.Rack;

/**
 * Implements actions defined in ChassisService
 *
 */
public class ChassisServiceImpl implements ChassisService {
   private static final Log _logger = LogFactory.getLog(ChassisServiceImpl.class);

   private final ModelObjectUriResolver _uriResolver;
   private final ObjectStore _objectStore;

   /**
    * Constructor.
    *
    * @param uriResolver
    *    Custom type resolver for ModelObject resources used in this sample
    *
    * @param objectStore The in-memory store used for this sample.
    */
   public ChassisServiceImpl(
            ModelObjectUriResolver uriResolver,
            ObjectStore objectStore) {
      _uriResolver = uriResolver;
      _objectStore = objectStore;
   }

   /* (non-Javadoc)
    * @see com.vmware.samples.chassisRackVSphere.ChassisService#createChassis()
    */
   @Override
   public URI createChassis(ChassisInfo chassisInfo, URI rackRef) {
      Rack rack = null;
      if (rackRef != null) {
         String uid = _uriResolver.getUid(rackRef);
         rack = _objectStore.getRack(uid);
         if (rack == null) {
            throw new IllegalArgumentException("Rack not found during create chassis operation: " + uid);
         }
      }
      Chassis chassis = _objectStore.createChassis(chassisInfo, rack, true);

      if (chassis == null) {
         _logger.info("Chassis not created because this name is taken: " + chassisInfo.name);
         return null;
      }
      _logger.info("Chassis created: " + chassis.toString());

      // Important: the returned object is the reference generated by the reference service
      // not the chassis object itself!  In practice it will be a URI that get mapped into
      // a IResourceReference in the UI layer.
      return chassis.getUri(_uriResolver);
   }

   /* (non-Javadoc)
    * @see com.vmware.samples.chassisservice.ChassisService#editChassis(java.lang.Object)
    */
   @Override
   public boolean editChassis(URI chassisRef) {
      String uid = _uriResolver.getUid(chassisRef);

      Chassis chassis = _objectStore.getChassis(uid);
      if (chassis == null) {
         // Chassis may have already been deleted by another user
         _logger.info("Chassis not found during edit operation: " + uid);
         return false;
      }

      // Extend the serverType value as a way to show a modified chassis in the UI
      ChassisInfo newInfo = new ChassisInfo(new String[] {chassis.getName(),
            (chassis.getServerType() + "_X"), chassis.getDimensions() });

      return _objectStore.replaceChassis(uid, newInfo);
   }

   /* (non-Javadoc)
    * @see com.vmware.samples.chassisservice.ChassisService#deleteChassis(java.lang.Object)
    */
   @Override
   public boolean deleteChassis(URI chassisRef) {
      String uid = _uriResolver.getUid(chassisRef);

      Chassis chassis = _objectStore.removeChassis(uid);
      if (chassis == null) {
         // Chassis may have already been deleted by another user
         _logger.info("Chassis not found during delete operation: " + uid);
         return false;
      }
      return true;
   }

   /* (non-Javadoc)
    * @see com.vmware.samples.chassisRackVSphere.ChassisService#addHost(java.lang.Object, java.lang.Object)
    */
   @Override
   public boolean addHost(Object hostRef, URI chassisRef) {
      String uid = _uriResolver.getUid(chassisRef);
      return _objectStore.addHostToChassis(hostRef, uid);
    }

}
